# Blünek Feedback System - Self-Hosted

A production-ready feedback collection system with multi-product support, webhooks, and admin dashboard.

## Features

- **Multi-Product Support**: Each product gets its own API key and feedback isolation
- **Webhook Delivery**: Real-time feedback delivery to your endpoints
- **Admin Dashboard**: Web-based management interface
- **Self-Contained**: SQLite database, no external dependencies
- **Docker Ready**: Includes Dockerfile and docker-compose.yml
- **Security**: Rate limiting, spam protection, input validation

## 🚀 Quick Start (5 minutes)

### Option 1: Direct Python

```bash
# 1. Install dependencies
pip install -r requirements.txt

# 2. Configure (edit config.yaml)
cp config.yaml.example config.yaml

# 3. Run migration
python migrate_db.py

# 4. Start server
python main.py
```

### Option 2: Docker

```bash
# 1. Configure
cp config.yaml.example config.yaml

# 2. Start with Docker Compose
docker-compose up -d
```

## 📋 Configuration

Edit `config.yaml`:

```yaml
# Change the admin API key!
security:
  admin_api_key: "your-secure-random-string-here"
  
# Set your domain for production
trusted_hosts: ["yourdomain.com", "localhost"]
cors:
  allowed_origins: ["https://yourdomain.com"]
```

## 🔧 Setup Products

1. **Open admin dashboard**: http://localhost:8310/admin.html
2. **Register products**: Get API keys for each product
3. **Configure webhooks**: (Optional) Set webhook URLs
4. **Embed widget**: Add to your websites

## 📝 Widget Integration

Add to any webpage:

```html
<!-- Load widget -->
<script src="http://localhost:8310/widget.js" data-product-key="YOUR_API_KEY"></script>

<!-- Manual integration -->
<div id="feedback-widget" data-product-key="YOUR_API_KEY"></div>
<script src="http://localhost:8310/widget.js"></script>
```

## 🔗 API Endpoints

### Products
- `POST /api/products/register` - Register new product
- `GET /api/products/{key}/feedback` - Get product feedback
- `PUT /api/products/{key}/webhook` - Configure webhook

### Feedback
- `POST /api/feedback` - Submit feedback
- `GET /api/feedback` - List all feedback (admin)
- `DELETE /api/feedback/{id}` - Delete feedback (admin)

### Admin
- `GET /admin.html` - Admin dashboard
- `GET /onboard` - Onboarding flow

## 🔒 Security

- **Rate Limiting**: 5 submissions per hour per IP
- **Input Validation**: All inputs sanitized and validated
- **Spam Protection**: Honeypot and timing checks
- **API Keys**: Secure random generation
- **Headers**: Security headers automatically added

## 📊 Webhook Format

When feedback is received, POST to your webhook URL:

```json
{
  "type": "bug",
  "message": "The submit button doesn't work",
  "email": "user@example.com",
  "product": "My App",
  "timestamp": "2024-02-05T21:30:00Z"
}
```

## 🗄️ Database

- **SQLite**: Single file database
- **Auto-migration**: Runs automatically on startup
- **Backup**: Simply copy the .db file

## 📁 File Structure

```
├── main.py              # Main application
├── models.py            # Database models
├── database.py          # Database setup
├── security.py          # Security utilities
├── webhooks.py          # Webhook delivery
├── config.yaml          # Configuration
├── requirements.txt     # Python dependencies
├── Dockerfile          # Docker build
├── docker-compose.yml  # Docker compose
├── static/             # Web assets
│   ├── admin.html      # Admin dashboard
│   ├── widget.js       # Feedback widget
│   └── demo.html       # Demo page
└── feedback.db         # SQLite database (created on startup)
```

## 🔧 Advanced Configuration

### Custom Database Path

```yaml
database:
  path: "/data/feedback.db"
```

### Webhook-Only Mode

Configure products to only send webhooks without storing in database:

```bash
curl -X PUT http://localhost:8310/api/products/YOUR_KEY/webhook \
  -H "Content-Type: application/json" \
  -d '{"webhook_url": "https://yourapp.com/webhook", "webhook_only": true}'
```

### Rate Limiting

```yaml
rate_limiting:
  max_requests_per_hour: 10  # Adjust as needed
```

## 🛠️ Troubleshooting

### Port Already in Use
```bash
# Change port in config.yaml
server:
  port: 8311
```

### Database Issues
```bash
# Reset database
rm feedback.db
python migrate_db.py
```

### Docker Permission Issues
```bash
# Fix permissions
sudo chown -R $USER:$USER ./data
```

## 📈 Scaling

For high traffic:
1. Use PostgreSQL instead of SQLite
2. Add Redis for rate limiting
3. Load balance multiple instances
4. Use proper reverse proxy (nginx)

## 🤝 Support

- **Issues**: Check logs in Docker: `docker-compose logs`
- **Database**: Backup regularly: `cp feedback.db backup-$(date +%Y%m%d).db`
- **Updates**: Pull new versions and restart

---

**Production Ready** ✅ Rate limiting ✅ Security headers ✅ Input validation ✅ Error handling
