import time
import os
import asyncio
import tempfile
import shutil
import zipfile
from datetime import datetime
from typing import List, Optional
from fastapi import FastAPI, HTTPException, Depends, Request, Header, Query
from fastapi.responses import HTMLResponse, FileResponse, JSONResponse, StreamingResponse
from fastapi.middleware.cors import CORSMiddleware
from fastapi.middleware.trustedhost import TrustedHostMiddleware
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from fastapi.staticfiles import StaticFiles
from sqlalchemy.orm import Session
from sqlalchemy import desc, and_
from pydantic import BaseModel

from models import Feedback, Product
from database import get_db, init_db
from security import (
    FeedbackSubmission, rate_limiter, check_honeypot, check_timing, 
    verify_formprotect, verify_admin_key, get_client_ip, sanitize_text
)
from webhooks import webhook_delivery
from encryption import ServerEncryption, ClientEncryption, is_encrypted_field, is_client_encrypted_field
import json

# Pydantic models for API
class ProductRegistration(BaseModel):
    name: str

class WebhookConfig(BaseModel):
    webhook_url: Optional[str] = None
    webhook_only: bool = False

# Initialize FastAPI app
app = FastAPI(
    title="Blünek Feedback System",
    description="Centralized feedback collection for all blünek products",
    version="2.0.0"
)

# Security headers middleware
@app.middleware("http")
async def add_security_headers(request: Request, call_next):
    response = await call_next(request)
    
    # Security headers
    response.headers["X-Content-Type-Options"] = "nosniff"
    response.headers["X-Frame-Options"] = "DENY"
    response.headers["X-XSS-Protection"] = "1; mode=block"
    response.headers["Strict-Transport-Security"] = "max-age=31536000; includeSubDomains"
    response.headers["Content-Security-Policy"] = (
        "default-src 'self'; "
        "script-src 'self' 'unsafe-inline' forms.blunek.services; "
        "style-src 'self' 'unsafe-inline'; "
        "connect-src 'self' forms.blunek.services; "
        "img-src 'self' data:; "
        "frame-src 'none'"
    )
    
    return response

# CORS configuration
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],  # In production, specify actual domains
    allow_credentials=False,
    allow_methods=["GET", "POST", "DELETE"],
    allow_headers=["*"],
)

# Trusted host middleware
app.add_middleware(
    TrustedHostMiddleware,
    allowed_hosts=["feedback.blunek.services", "localhost", "127.0.0.1"]
)

# Initialize database on startup
# Helper functions
def get_product_by_key(db: Session, api_key: str) -> Optional[Product]:
    """Get product by API key"""
    return db.query(Product).filter(Product.api_key == api_key).first()

@app.on_event("startup")
async def startup_event():
    init_db()
    print("Enhanced Feedback system started on port 8310")

# Health check endpoint
@app.get("/health")
async def health_check():
    return {"status": "ok", "timestamp": time.time()}

# Submit feedback endpoint (enhanced)
@app.post("/api/feedback")
async def submit_feedback(
    feedback: FeedbackSubmission,
    request: Request,
    db: Session = Depends(get_db)
):
    client_ip = get_client_ip(request)
    submit_time = time.time()
    
    # Rate limiting
    if not rate_limiter.is_allowed(client_ip):
        raise HTTPException(
            status_code=429, 
            detail="Rate limit exceeded. Maximum 5 submissions per hour."
        )
    
    # Honeypot check
    if not check_honeypot(feedback.honeypot):
        raise HTTPException(status_code=400, detail="Invalid submission")
    
    # Timing check
    if not check_timing(submit_time, feedback.render_time):
        raise HTTPException(status_code=400, detail="Submission too fast")
    
    # FormProtect verification
    if feedback.formprotect_token:
        if not await verify_formprotect(feedback.formprotect_token, client_ip):
            raise HTTPException(status_code=400, detail="Security verification failed")
    
    # Validate and clean input
    try:
        cleaned_data = feedback.validate_and_clean()
    except HTTPException:
        raise
    except Exception as e:
        raise HTTPException(status_code=400, detail="Invalid input data")
    
    # Handle product key if provided
    product = None
    if feedback.product_key:
        product = get_product_by_key(db, feedback.product_key)
        if not product:
            raise HTTPException(status_code=400, detail="Invalid product key")
    
    # Prepare feedback data for webhook
    feedback_data = {
        "type": cleaned_data["type"],
        "message": cleaned_data["message"],
        "email": cleaned_data["email"],
        "product": cleaned_data["product"]
    }
    
    # Send webhook if configured
    webhook_sent = False
    if product and product.webhook_url:
        webhook_sent = await webhook_delivery.deliver_feedback_webhook(product, feedback_data)
    
    # Store in database unless webhook-only mode
    feedback_id = None
    if not (product and product.webhook_only and webhook_sent):
        # Handle encryption based on product settings
        message_to_store = cleaned_data["message"]
        email_to_store = cleaned_data["email"]
        is_client_encrypted = False
        
        if product:
            # Check if this is client-encrypted data
            if (product.client_encryption and 
                cleaned_data["message"].startswith("CLIENT_ENC:")):
                # Client-side encrypted - store as-is, we cannot decrypt
                is_client_encrypted = True
                
            elif product.encryption_enabled and not is_client_encrypted:
                # Server-side encryption - encrypt before storing (Protected mode)
                # Always encrypt emails if provided, to comply with privacy requirements
                encrypted_fields = ServerEncryption.encrypt_feedback_fields(
                    cleaned_data["message"], 
                    cleaned_data["email"], 
                    product.api_key
                )
                message_to_store = json.dumps(encrypted_fields["message"])
                email_to_store = json.dumps(encrypted_fields["email"])
            
            # For basic mode (no encryption), ensure no email is accidentally stored in plain text
            elif not product.encryption_enabled and not product.client_encryption:
                # Basic mode: feedback text only, no email should be provided
                if cleaned_data["email"]:
                    raise HTTPException(status_code=400, detail="Email not allowed in basic privacy mode")
                message_to_store = cleaned_data["message"]  # Store message as-is
                email_to_store = ""  # Never store email in basic mode
        
        # Create feedback entry
        db_feedback = Feedback(
            type=cleaned_data["type"],
            product=cleaned_data["product"],
            message=message_to_store,
            email=email_to_store,
            product_id=product.id if product else None,
            encrypted_client=is_client_encrypted,
            ip_address=client_ip,
            user_agent=request.headers.get("User-Agent", "")[:500]  # Limit length
        )
        
        try:
            db.add(db_feedback)
            db.commit()
            db.refresh(db_feedback)
            feedback_id = db_feedback.id
        except Exception as e:
            db.rollback()
            raise HTTPException(status_code=500, detail="Failed to save feedback")
    
    return {
        "success": True,
        "message": "Thank you for your feedback!",
        "id": feedback_id,
        "webhook_delivered": webhook_sent
    }

# Admin: List feedback
@app.get("/api/feedback")
async def list_feedback(
    request: Request,
    db: Session = Depends(get_db),
    api_key: Optional[str] = Header(None, alias="X-API-Key"),
    type_filter: Optional[str] = Query(None, description="Filter by type"),
    product_filter: Optional[str] = Query(None, description="Filter by product"),
    spam_filter: Optional[bool] = Query(None, description="Filter by spam status"),
    reviewed_filter: Optional[bool] = Query(None, description="Filter by reviewed status"),
    limit: int = Query(50, le=200, description="Max results"),
    offset: int = Query(0, ge=0, description="Offset for pagination")
):
    # Verify admin API key
    if not verify_admin_key(api_key):
        raise HTTPException(status_code=401, detail="Invalid API key")
    
    # Build query with filters
    query = db.query(Feedback)
    
    if type_filter:
        query = query.filter(Feedback.type == type_filter)
    if product_filter:
        query = query.filter(Feedback.product.ilike(f"%{product_filter}%"))
    if spam_filter is not None:
        query = query.filter(Feedback.is_spam == spam_filter)
    if reviewed_filter is not None:
        query = query.filter(Feedback.is_reviewed == reviewed_filter)
    
    # Order by newest first and paginate
    feedback_list = query.order_by(desc(Feedback.created_at)).offset(offset).limit(limit).all()
    total_count = query.count()
    
    # Format response with decryption
    feedback_data = []
    for item in feedback_list:
        message = item.message
        email = item.email
        encryption_status = "none"
        
        # Handle decryption if needed
        if item.product_obj:
            if item.encrypted_client:
                # Client-encrypted data - we cannot decrypt
                encryption_status = "client_encrypted"
                message = "[CLIENT ENCRYPTED - CANNOT DECRYPT]"
                email = "[CLIENT ENCRYPTED - CANNOT DECRYPT]" if email else None
            elif item.product_obj.encryption_enabled:
                # Server-encrypted data - we can decrypt
                try:
                    encrypted_message = json.loads(item.message) if item.message else {}
                    encrypted_email = json.loads(item.email) if item.email else {}
                    
                    if is_encrypted_field(encrypted_message):
                        decrypted_fields = ServerEncryption.decrypt_feedback_fields(
                            encrypted_message,
                            encrypted_email,
                            item.product_obj.api_key
                        )
                        message = decrypted_fields["message"]
                        email = decrypted_fields["email"]
                        encryption_status = "server_encrypted"
                except (json.JSONDecodeError, Exception) as e:
                    # Fallback to raw data if decryption fails
                    encryption_status = "decrypt_error"
        
        feedback_data.append({
            "id": item.id,
            "type": item.type,
            "product": item.product,
            "message": message,
            "email": email,
            "is_spam": item.is_spam,
            "is_reviewed": item.is_reviewed,
            "encrypted_client": getattr(item, 'encrypted_client', False),
            "encryption_status": encryption_status,
            "created_at": item.created_at.isoformat(),
            "reviewed_at": item.reviewed_at.isoformat() if item.reviewed_at else None,
            "ip_address": item.ip_address[-8:] + "..." if item.ip_address else None  # Masked for privacy
        })
    
    return {
        "feedback": feedback_data,
        "total": total_count,
        "offset": offset,
        "limit": limit
    }

# Admin: Delete feedback
@app.delete("/api/feedback/{feedback_id}")
async def delete_feedback(
    feedback_id: int,
    request: Request,
    db: Session = Depends(get_db),
    api_key: Optional[str] = Header(None, alias="X-API-Key")
):
    # Verify admin API key
    if not verify_admin_key(api_key):
        raise HTTPException(status_code=401, detail="Invalid API key")
    
    # Find feedback
    feedback = db.query(Feedback).filter(Feedback.id == feedback_id).first()
    if not feedback:
        raise HTTPException(status_code=404, detail="Feedback not found")
    
    # Delete feedback
    try:
        db.delete(feedback)
        db.commit()
        return {"success": True, "message": "Feedback deleted"}
    except Exception as e:
        db.rollback()
        raise HTTPException(status_code=500, detail="Failed to delete feedback")

# Admin: Mark feedback as spam/reviewed
@app.patch("/api/feedback/{feedback_id}")
async def update_feedback(
    feedback_id: int,
    request: Request,
    db: Session = Depends(get_db),
    api_key: Optional[str] = Header(None, alias="X-API-Key"),
    mark_spam: Optional[bool] = Query(None, description="Mark as spam"),
    mark_reviewed: Optional[bool] = Query(None, description="Mark as reviewed")
):
    # Verify admin API key
    if not verify_admin_key(api_key):
        raise HTTPException(status_code=401, detail="Invalid API key")
    
    # Find feedback
    feedback = db.query(Feedback).filter(Feedback.id == feedback_id).first()
    if not feedback:
        raise HTTPException(status_code=404, detail="Feedback not found")
    
    # Update fields
    try:
        if mark_spam is not None:
            feedback.is_spam = mark_spam
        if mark_reviewed is not None:
            feedback.is_reviewed = mark_reviewed
            if mark_reviewed:
                feedback.reviewed_at = datetime.utcnow()
        
        db.commit()
        return {"success": True, "message": "Feedback updated"}
    except Exception as e:
        db.rollback()
        raise HTTPException(status_code=500, detail="Failed to update feedback")

# ====================== NEW PRODUCT MANAGEMENT ENDPOINTS ======================

# Register new product
@app.post("/api/products/register")
async def register_product(
    product_data: ProductRegistration,
    db: Session = Depends(get_db)
):
    """Register a new product and get API key"""
    try:
        # Generate unique API key
        api_key = Product.generate_api_key()
        
        # Create product with Protected mode as default (encryption enabled)
        new_product = Product(
            name=product_data.name.strip(),
            api_key=api_key,
            encryption_enabled=True  # Protected mode is now the default
        )
        
        db.add(new_product)
        db.commit()
        db.refresh(new_product)
        
        return {
            "success": True,
            "product_id": new_product.id,
            "name": new_product.name,
            "api_key": api_key,
            "message": f"Product '{new_product.name}' registered successfully!"
        }
    except Exception as e:
        db.rollback()
        raise HTTPException(status_code=500, detail=f"Failed to register product: {str(e)}")

# Get feedback for specific product
@app.get("/api/products/{api_key}/feedback")
async def get_product_feedback(
    api_key: str,
    db: Session = Depends(get_db),
    type_filter: Optional[str] = Query(None, description="Filter by type"),
    spam_filter: Optional[bool] = Query(None, description="Filter by spam status"),
    limit: int = Query(50, le=200, description="Max results"),
    offset: int = Query(0, ge=0, description="Offset for pagination")
):
    """Get feedback for a specific product"""
    # Verify product exists
    product = get_product_by_key(db, api_key)
    if not product:
        raise HTTPException(status_code=404, detail="Product not found")
    
    # Build query with filters
    query = db.query(Feedback).filter(Feedback.product_id == product.id)
    
    if type_filter:
        query = query.filter(Feedback.type == type_filter)
    if spam_filter is not None:
        query = query.filter(Feedback.is_spam == spam_filter)
    
    # Order by newest first and paginate
    feedback_list = query.order_by(desc(Feedback.created_at)).offset(offset).limit(limit).all()
    total_count = query.count()
    
    # Format response
    feedback_data = []
    for item in feedback_list:
        feedback_data.append({
            "id": item.id,
            "type": item.type,
            "message": item.message,
            "email": item.email,
            "is_spam": item.is_spam,
            "is_reviewed": item.is_reviewed,
            "created_at": item.created_at.isoformat(),
            "reviewed_at": item.reviewed_at.isoformat() if item.reviewed_at else None
        })
    
    return {
        "product": product.name,
        "feedback": feedback_data,
        "total": total_count,
        "offset": offset,
        "limit": limit
    }

# Configure webhook for product
@app.put("/api/products/{api_key}/webhook")
async def configure_webhook(
    api_key: str,
    webhook_config: WebhookConfig,
    db: Session = Depends(get_db)
):
    """Configure webhook URL for a product"""
    # Verify product exists
    product = get_product_by_key(db, api_key)
    if not product:
        raise HTTPException(status_code=404, detail="Product not found")
    
    try:
        # Update webhook configuration
        product.webhook_url = webhook_config.webhook_url
        product.webhook_only = webhook_config.webhook_only
        
        db.commit()
        
        return {
            "success": True,
            "message": "Webhook configuration updated",
            "webhook_url": product.webhook_url,
            "webhook_only": product.webhook_only
        }
    except Exception as e:
        db.rollback()
        raise HTTPException(status_code=500, detail=f"Failed to update webhook: {str(e)}")

# ====================== ENCRYPTION MANAGEMENT ======================

class EncryptionConfig(BaseModel):
    encryption_enabled: bool = False
    client_encryption: bool = False
    client_encryption_key: Optional[str] = None

class PrivacyConfig(BaseModel):
    privacy_mode: str  # "basic", "protected", "zero-knowledge"
    client_encryption_key: Optional[str] = None

@app.get("/api/products/{api_key}/encryption")
async def get_encryption_config(
    api_key: str,
    db: Session = Depends(get_db)
):
    """Get encryption configuration for a product"""
    product = get_product_by_key(db, api_key)
    if not product:
        raise HTTPException(status_code=404, detail="Product not found")
    
    return {
        "product_name": product.name,
        "encryption_enabled": product.encryption_enabled or False,
        "client_encryption": product.client_encryption or False,
        "has_client_key": bool(product.client_encryption_key_hash)
    }

@app.put("/api/products/{api_key}/encryption")
async def configure_encryption(
    api_key: str,
    encryption_config: EncryptionConfig,
    db: Session = Depends(get_db)
):
    """Configure encryption settings for a product"""
    product = get_product_by_key(db, api_key)
    if not product:
        raise HTTPException(status_code=404, detail="Product not found")
    
    try:
        # Validate configuration
        if encryption_config.client_encryption and encryption_config.encryption_enabled:
            raise HTTPException(
                status_code=400, 
                detail="Cannot enable both server and client encryption simultaneously"
            )
        
        # Update encryption settings
        product.encryption_enabled = encryption_config.encryption_enabled
        product.client_encryption = encryption_config.client_encryption
        
        # Handle client encryption key
        if encryption_config.client_encryption and encryption_config.client_encryption_key:
            product.client_encryption_key_hash = ClientEncryption.hash_encryption_key(
                encryption_config.client_encryption_key
            )
        elif not encryption_config.client_encryption:
            product.client_encryption_key_hash = None
        
        db.commit()
        
        return {
            "success": True,
            "message": "Encryption configuration updated",
            "encryption_enabled": product.encryption_enabled,
            "client_encryption": product.client_encryption,
            "has_client_key": bool(product.client_encryption_key_hash)
        }
    except HTTPException:
        raise
    except Exception as e:
        db.rollback()
        raise HTTPException(status_code=500, detail=f"Failed to update encryption: {str(e)}")

@app.put("/api/products/{api_key}/privacy")
async def configure_privacy_mode(
    api_key: str,
    privacy_config: PrivacyConfig,
    db: Session = Depends(get_db)
):
    """Configure privacy mode using user-friendly labels"""
    product = get_product_by_key(db, api_key)
    if not product:
        raise HTTPException(status_code=404, detail="Product not found")
    
    try:
        # Reset all encryption settings
        product.encryption_enabled = False
        product.client_encryption = False
        product.client_encryption_key_hash = None
        
        # Configure based on privacy mode
        if privacy_config.privacy_mode == "basic":
            # Basic: No encryption, no email collection
            pass  # All encryption flags remain False
            
        elif privacy_config.privacy_mode == "protected":
            # Protected: Server-side encryption (default)
            product.encryption_enabled = True
            
        elif privacy_config.privacy_mode == "zero-knowledge":
            # Zero-knowledge: Client-side encryption
            if not privacy_config.client_encryption_key:
                raise HTTPException(status_code=400, detail="Encryption key required for zero-knowledge mode")
            
            product.client_encryption = True
            product.client_encryption_key_hash = ClientEncryption.hash_encryption_key(
                privacy_config.client_encryption_key
            )
            
        else:
            raise HTTPException(status_code=400, detail="Invalid privacy mode")
        
        db.commit()
        
        return {
            "success": True,
            "privacy_mode": privacy_config.privacy_mode,
            "message": f"Privacy mode set to '{privacy_config.privacy_mode}'"
        }
    except HTTPException:
        raise
    except Exception as e:
        db.rollback()
        raise HTTPException(status_code=500, detail=f"Failed to configure privacy: {str(e)}")

@app.get("/privacy-options", response_class=HTMLResponse)
async def privacy_options():
    """Serve privacy options explanation page"""
    return FileResponse("static/privacy.html")

# ====================== SELF-HOSTING PACKAGE ======================

@app.get("/self-host/download")
async def download_self_host_package():
    """Download self-hosting package as ZIP"""
    try:
        # Create temporary directory
        with tempfile.TemporaryDirectory() as temp_dir:
            package_dir = os.path.join(temp_dir, "blunek-feedback-selfhost")
            os.makedirs(package_dir)
            
            # Copy essential files
            files_to_copy = [
                "main.py", "models.py", "database.py", "security.py", "webhooks.py",
                "requirements.txt", "migrate_db.py"
            ]
            
            for file_name in files_to_copy:
                if os.path.exists(file_name):
                    shutil.copy2(file_name, package_dir)
            
            # Copy static directory
            if os.path.exists("static"):
                shutil.copytree("static", os.path.join(package_dir, "static"))
            
            # Create config.yaml template
            config_content = """# Blünek Feedback System - Self-Hosted Configuration

# Database
database:
  path: "./feedback.db"  # SQLite database file path

# Server
server:
  host: "0.0.0.0"
  port: 8310

# Security
security:
  admin_api_key: "CHANGE_THIS_TO_A_SECURE_RANDOM_STRING"
  
# Optional: Rate limiting
rate_limiting:
  max_requests_per_hour: 5

# Optional: CORS settings (for production)
cors:
  allowed_origins: ["*"]  # Change to your domain(s)
  
# Optional: Trusted hosts
trusted_hosts: ["localhost", "127.0.0.1"]  # Add your domain
"""
            
            with open(os.path.join(package_dir, "config.yaml"), "w") as f:
                f.write(config_content)
            
            # Create Dockerfile
            dockerfile_content = """FROM python:3.9-slim

WORKDIR /app

# Copy requirements first for better Docker layer caching
COPY requirements.txt .
RUN pip install --no-cache-dir -r requirements.txt

# Copy application files
COPY . .

# Run migration on startup then start app
EXPOSE 8310
CMD ["sh", "-c", "python migrate_db.py && python main.py"]
"""
            
            with open(os.path.join(package_dir, "Dockerfile"), "w") as f:
                f.write(dockerfile_content)
            
            # Create docker-compose.yml
            compose_content = """version: '3.8'

services:
  feedback:
    build: .
    ports:
      - "8310:8310"
    volumes:
      - ./data:/app/data
      - ./config.yaml:/app/config.yaml
    environment:
      - PYTHONUNBUFFERED=1
    restart: unless-stopped

volumes:
  data:
"""
            
            with open(os.path.join(package_dir, "docker-compose.yml"), "w") as f:
                f.write(compose_content)
            
            # Create comprehensive README
            readme_content = """# Blünek Feedback System - Self-Hosted

A production-ready feedback collection system with multi-product support, webhooks, and admin dashboard.

## Features

- **Multi-Product Support**: Each product gets its own API key and feedback isolation
- **Webhook Delivery**: Real-time feedback delivery to your endpoints
- **Admin Dashboard**: Web-based management interface
- **Self-Contained**: SQLite database, no external dependencies
- **Docker Ready**: Includes Dockerfile and docker-compose.yml
- **Security**: Rate limiting, spam protection, input validation

## 🚀 Quick Start (5 minutes)

### Option 1: Direct Python

```bash
# 1. Install dependencies
pip install -r requirements.txt

# 2. Configure (edit config.yaml)
cp config.yaml.example config.yaml

# 3. Run migration
python migrate_db.py

# 4. Start server
python main.py
```

### Option 2: Docker

```bash
# 1. Configure
cp config.yaml.example config.yaml

# 2. Start with Docker Compose
docker-compose up -d
```

## 📋 Configuration

Edit `config.yaml`:

```yaml
# Change the admin API key!
security:
  admin_api_key: "your-secure-random-string-here"
  
# Set your domain for production
trusted_hosts: ["yourdomain.com", "localhost"]
cors:
  allowed_origins: ["https://yourdomain.com"]
```

## 🔧 Setup Products

1. **Open admin dashboard**: http://localhost:8310/admin.html
2. **Register products**: Get API keys for each product
3. **Configure webhooks**: (Optional) Set webhook URLs
4. **Embed widget**: Add to your websites

## 📝 Widget Integration

Add to any webpage:

```html
<!-- Load widget -->
<script src="http://localhost:8310/widget.js" data-product-key="YOUR_API_KEY"></script>

<!-- Manual integration -->
<div id="feedback-widget" data-product-key="YOUR_API_KEY"></div>
<script src="http://localhost:8310/widget.js"></script>
```

## 🔗 API Endpoints

### Products
- `POST /api/products/register` - Register new product
- `GET /api/products/{key}/feedback` - Get product feedback
- `PUT /api/products/{key}/webhook` - Configure webhook

### Feedback
- `POST /api/feedback` - Submit feedback
- `GET /api/feedback` - List all feedback (admin)
- `DELETE /api/feedback/{id}` - Delete feedback (admin)

### Admin
- `GET /admin.html` - Admin dashboard
- `GET /onboard` - Onboarding flow

## 🔒 Security

- **Rate Limiting**: 5 submissions per hour per IP
- **Input Validation**: All inputs sanitized and validated
- **Spam Protection**: Honeypot and timing checks
- **API Keys**: Secure random generation
- **Headers**: Security headers automatically added

## 📊 Webhook Format

When feedback is received, POST to your webhook URL:

```json
{
  "type": "bug",
  "message": "The submit button doesn't work",
  "email": "user@example.com",
  "product": "My App",
  "timestamp": "2024-02-05T21:30:00Z"
}
```

## 🗄️ Database

- **SQLite**: Single file database
- **Auto-migration**: Runs automatically on startup
- **Backup**: Simply copy the .db file

## 📁 File Structure

```
├── main.py              # Main application
├── models.py            # Database models
├── database.py          # Database setup
├── security.py          # Security utilities
├── webhooks.py          # Webhook delivery
├── config.yaml          # Configuration
├── requirements.txt     # Python dependencies
├── Dockerfile          # Docker build
├── docker-compose.yml  # Docker compose
├── static/             # Web assets
│   ├── admin.html      # Admin dashboard
│   ├── widget.js       # Feedback widget
│   └── demo.html       # Demo page
└── feedback.db         # SQLite database (created on startup)
```

## 🔧 Advanced Configuration

### Custom Database Path

```yaml
database:
  path: "/data/feedback.db"
```

### Webhook-Only Mode

Configure products to only send webhooks without storing in database:

```bash
curl -X PUT http://localhost:8310/api/products/YOUR_KEY/webhook \\
  -H "Content-Type: application/json" \\
  -d '{"webhook_url": "https://yourapp.com/webhook", "webhook_only": true}'
```

### Rate Limiting

```yaml
rate_limiting:
  max_requests_per_hour: 10  # Adjust as needed
```

## 🛠️ Troubleshooting

### Port Already in Use
```bash
# Change port in config.yaml
server:
  port: 8311
```

### Database Issues
```bash
# Reset database
rm feedback.db
python migrate_db.py
```

### Docker Permission Issues
```bash
# Fix permissions
sudo chown -R $USER:$USER ./data
```

## 📈 Scaling

For high traffic:
1. Use PostgreSQL instead of SQLite
2. Add Redis for rate limiting
3. Load balance multiple instances
4. Use proper reverse proxy (nginx)

## 🤝 Support

- **Issues**: Check logs in Docker: `docker-compose logs`
- **Database**: Backup regularly: `cp feedback.db backup-$(date +%Y%m%d).db`
- **Updates**: Pull new versions and restart

---

**Production Ready** ✅ Rate limiting ✅ Security headers ✅ Input validation ✅ Error handling
"""
            
            with open(os.path.join(package_dir, "README.md"), "w") as f:
                f.write(readme_content)
            
            # Create ZIP file
            zip_path = os.path.join(temp_dir, "blunek-feedback-selfhost.zip")
            with zipfile.ZipFile(zip_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
                for root, dirs, files in os.walk(package_dir):
                    for file in files:
                        file_path = os.path.join(root, file)
                        arc_name = os.path.relpath(file_path, package_dir)
                        zipf.write(file_path, arc_name)
            
            # Read ZIP file into memory and return it
            with open(zip_path, "rb") as f:
                zip_data = f.read()
            
            return StreamingResponse(
                iter([zip_data]),
                media_type="application/zip",
                headers={"Content-Disposition": "attachment; filename=blunek-feedback-selfhost.zip"}
            )
    
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Failed to create package: {str(e)}")

# ====================== ONBOARDING FLOW ======================

@app.get("/onboard", response_class=HTMLResponse)
async def onboarding_page():
    """Serve onboarding page"""
    return FileResponse("static/onboard.html")

# Serve widget JavaScript
@app.get("/widget.js")
async def get_widget():
    return FileResponse(
        "static/widget.js",
        media_type="application/javascript",
        headers={"Cache-Control": "public, max-age=3600"}
    )

# Serve static files
app.mount("/static", StaticFiles(directory="static"), name="static")

if __name__ == "__main__":
    import uvicorn
    uvicorn.run("main:app", host="0.0.0.0", port=8310, reload=True)
# Landing page
@app.get("/", response_class=HTMLResponse)
async def landing():
    """Serve landing page"""
    with open("static/demo.html", "r") as f:
        return f.read()
