#!/usr/bin/env python3
"""
Database migration script for blünek Feedback System
Adds products table and product_id column to feedback table
"""

import sqlite3
import os
from datetime import datetime

DB_PATH = os.path.join(os.path.dirname(__file__), "feedback.db")

def migrate_database():
    """Run database migration"""
    print("Starting database migration...")
    
    if not os.path.exists(DB_PATH):
        print(f"Database file not found at {DB_PATH}")
        print("This is fine for new installations - tables will be created on first run.")
        return
    
    # Connect to database
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()
    
    try:
        # Check if products table exists
        cursor.execute("""
            SELECT name FROM sqlite_master 
            WHERE type='table' AND name='products'
        """)
        
        if not cursor.fetchone():
            print("Creating products table...")
            cursor.execute("""
                CREATE TABLE products (
                    id INTEGER PRIMARY KEY,
                    name VARCHAR(100) NOT NULL,
                    api_key VARCHAR(64) UNIQUE NOT NULL,
                    webhook_url VARCHAR(500),
                    webhook_only BOOLEAN DEFAULT 0,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            """)
            print("✓ Products table created")
        else:
            print("✓ Products table already exists")
        
        # Check if product_id column exists in feedback table
        cursor.execute("PRAGMA table_info(feedback)")
        columns = [column[1] for column in cursor.fetchall()]
        
        if 'product_id' not in columns:
            print("Adding product_id column to feedback table...")
            cursor.execute("""
                ALTER TABLE feedback 
                ADD COLUMN product_id INTEGER 
                REFERENCES products(id)
            """)
            
            # Create index on product_id
            cursor.execute("""
                CREATE INDEX ix_feedback_product_id 
                ON feedback(product_id)
            """)
            print("✓ product_id column added with index")
        else:
            print("✓ product_id column already exists")
        
        # Check and add encryption columns to products table
        cursor.execute("PRAGMA table_info(products)")
        product_columns = [column[1] for column in cursor.fetchall()]
        
        if 'encryption_enabled' not in product_columns:
            print("Adding encryption_enabled column to products table...")
            cursor.execute("ALTER TABLE products ADD COLUMN encryption_enabled BOOLEAN DEFAULT 0")
            print("✓ encryption_enabled column added")
        
        if 'client_encryption' not in product_columns:
            print("Adding client_encryption column to products table...")
            cursor.execute("ALTER TABLE products ADD COLUMN client_encryption BOOLEAN DEFAULT 0")
            print("✓ client_encryption column added")
        
        if 'client_encryption_key_hash' not in product_columns:
            print("Adding client_encryption_key_hash column to products table...")
            cursor.execute("ALTER TABLE products ADD COLUMN client_encryption_key_hash VARCHAR(64)")
            print("✓ client_encryption_key_hash column added")
        
        # Check and add encrypted_client column to feedback table
        if 'encrypted_client' not in columns:
            print("Adding encrypted_client column to feedback table...")
            cursor.execute("ALTER TABLE feedback ADD COLUMN encrypted_client BOOLEAN DEFAULT 0")
            print("✓ encrypted_client column added")
        
        # Create a default "Legacy" product for existing feedback if needed
        cursor.execute("SELECT COUNT(*) FROM feedback WHERE product_id IS NULL")
        unassigned_count = cursor.fetchone()[0]
        
        if unassigned_count > 0:
            print(f"Found {unassigned_count} feedback items without product_id")
            
            # Check if legacy product exists
            cursor.execute("SELECT id FROM products WHERE name = 'Legacy Feedback'")
            legacy_product = cursor.fetchone()
            
            if not legacy_product:
                # Import here to avoid circular imports
                import secrets
                import string
                alphabet = string.ascii_letters + string.digits
                api_key = ''.join(secrets.choice(alphabet) for _ in range(48))
                
                cursor.execute("""
                    INSERT INTO products (name, api_key, created_at)
                    VALUES ('Legacy Feedback', ?, ?)
                """, (api_key, datetime.utcnow()))
                
                legacy_product_id = cursor.lastrowid
                print(f"✓ Created Legacy product with API key: {api_key}")
            else:
                legacy_product_id = legacy_product[0]
                print("✓ Using existing Legacy product")
            
            # Assign legacy product to unassigned feedback
            cursor.execute("""
                UPDATE feedback 
                SET product_id = ? 
                WHERE product_id IS NULL
            """, (legacy_product_id,))
            
            print(f"✓ Assigned {unassigned_count} legacy feedback items to Legacy product")
        
        # Commit all changes
        conn.commit()
        print("✅ Database migration completed successfully!")
        
        # Show summary
        cursor.execute("SELECT COUNT(*) FROM products")
        product_count = cursor.fetchone()[0]
        
        cursor.execute("SELECT COUNT(*) FROM feedback")
        feedback_count = cursor.fetchone()[0]
        
        print(f"\nSummary:")
        print(f"- Products: {product_count}")
        print(f"- Feedback items: {feedback_count}")
        
    except Exception as e:
        print(f"❌ Migration failed: {e}")
        conn.rollback()
        raise
    finally:
        conn.close()

if __name__ == "__main__":
    migrate_database()