from sqlalchemy import Column, Integer, String, DateTime, Text, Boolean, ForeignKey
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.sql import func
from sqlalchemy.orm import relationship
from datetime import datetime
import secrets
import string

Base = declarative_base()

class Product(Base):
    __tablename__ = "products"
    
    id = Column(Integer, primary_key=True, index=True)
    name = Column(String(100), nullable=False)
    api_key = Column(String(64), unique=True, nullable=False, index=True)
    webhook_url = Column(String(500), nullable=True)
    webhook_only = Column(Boolean, default=False)  # If True, don't store feedback in DB
    encryption_enabled = Column(Boolean, default=False)  # Server-side encryption
    client_encryption = Column(Boolean, default=False)  # Client-side encryption
    client_encryption_key_hash = Column(String(64), nullable=True)  # Hash of client encryption key
    created_at = Column(DateTime, server_default=func.now())
    
    # Relationship to feedback
    feedback_items = relationship("Feedback", back_populates="product_obj")
    
    def __repr__(self):
        return f"<Product(id={self.id}, name={self.name}, api_key={self.api_key[:8]}...)>"
    
    @staticmethod
    def generate_api_key():
        """Generate a secure random API key"""
        alphabet = string.ascii_letters + string.digits
        return ''.join(secrets.choice(alphabet) for _ in range(48))

class Feedback(Base):
    __tablename__ = "feedback"
    
    id = Column(Integer, primary_key=True, index=True)
    type = Column(String(20), nullable=False)  # bug, feature, suggestion
    product = Column(String(100), nullable=True)  # which blünek product (legacy)
    product_id = Column(Integer, ForeignKey("products.id"), nullable=True, index=True)
    message = Column(Text(2000), nullable=False)  # feedback message
    email = Column(String(255), nullable=True)  # optional for follow-up
    ip_address = Column(String(45), nullable=False)  # for rate limiting
    user_agent = Column(String(500), nullable=True)  # for spam detection
    is_spam = Column(Boolean, default=False)  # admin marking
    is_reviewed = Column(Boolean, default=False)  # admin marking
    encrypted_client = Column(Boolean, default=False)  # True if data is client-encrypted
    created_at = Column(DateTime, server_default=func.now())
    reviewed_at = Column(DateTime, nullable=True)
    
    # Relationship to product
    product_obj = relationship("Product", back_populates="feedback_items")
    
    def __repr__(self):
        return f"<Feedback(id={self.id}, type={self.type}, product={self.product})>"