import os
import re
import time
import hmac
import hashlib
from typing import Optional, Dict, Any
from datetime import datetime, timedelta
from collections import defaultdict, deque
import bleach
from email_validator import validate_email, EmailNotValidError
from fastapi import HTTPException, Request
from pydantic import BaseModel
import httpx

# Environment variables
ADMIN_API_KEY = os.getenv("ADMIN_API_KEY", "your-secure-admin-key-here")
FORMPROTECT_SECRET = os.getenv("FORMPROTECT_SECRET", "your-formprotect-secret")
FORMPROTECT_URL = "https://forms.blunek.services"

# Rate limiting - in-memory store (use Redis in production)
class RateLimiter:
    def __init__(self):
        self.requests: Dict[str, deque] = defaultdict(deque)
        self.max_requests = 5
        self.window_seconds = 3600  # 1 hour
        
    def is_allowed(self, ip: str) -> bool:
        now = time.time()
        # Clean old requests
        while self.requests[ip] and self.requests[ip][0] <= now - self.window_seconds:
            self.requests[ip].popleft()
            
        # Check if under limit
        if len(self.requests[ip]) >= self.max_requests:
            return False
            
        # Add current request
        self.requests[ip].append(now)
        return True

rate_limiter = RateLimiter()

def sanitize_text(text: str, max_length: int = None) -> str:
    """Sanitize user input - strip HTML, limit length"""
    if not text:
        return ""
    
    # Strip HTML tags and clean
    cleaned = bleach.clean(text, tags=[], strip=True)
    
    # Limit length
    if max_length and len(cleaned) > max_length:
        cleaned = cleaned[:max_length]
    
    return cleaned.strip()

def validate_email_address(email: str) -> Optional[str]:
    """Validate email format, return None if invalid"""
    if not email:
        return None
        
    try:
        # Normalize and validate
        valid = validate_email(email)
        return valid.email
    except EmailNotValidError:
        return None

def check_honeypot(honeypot_field: Optional[str]) -> bool:
    """Check if honeypot field is empty (bots usually fill it)"""
    return not honeypot_field or honeypot_field.strip() == ""

def check_timing(submit_time: float, render_time: Optional[float] = None) -> bool:
    """Check if form was submitted too quickly (< 3 seconds)"""
    if render_time is None:
        return True  # Skip timing check if no render time provided
        
    time_diff = submit_time - render_time
    return time_diff >= 3.0  # Must take at least 3 seconds

async def verify_formprotect(token: str, ip: str) -> bool:
    """Verify FormProtect token"""
    if not token:
        return False
        
    try:
        async with httpx.AsyncClient() as client:
            response = await client.post(
                f"{FORMPROTECT_URL}/verify",
                json={
                    "token": token,
                    "secret": FORMPROTECT_SECRET,
                    "ip": ip
                },
                timeout=5.0
            )
            return response.status_code == 200 and response.json().get("valid", False)
    except:
        # If FormProtect is down, log and allow submission
        # In production, you might want to be more strict
        return True

def verify_admin_key(api_key: Optional[str]) -> bool:
    """Verify admin API key"""
    if not api_key:
        return False
    return hmac.compare_digest(api_key, ADMIN_API_KEY)

def get_client_ip(request: Request) -> str:
    """Get client IP address, handling proxies"""
    # Check for forwarded IP first
    forwarded = request.headers.get("X-Forwarded-For")
    if forwarded:
        return forwarded.split(",")[0].strip()
    
    real_ip = request.headers.get("X-Real-IP")
    if real_ip:
        return real_ip
    
    return request.client.host

# Pydantic models for validation
class FeedbackSubmission(BaseModel):
    type: str
    product: Optional[str] = None
    message: str
    email: Optional[str] = None
    honeypot: Optional[str] = None  # Should be empty
    formprotect_token: Optional[str] = None
    render_time: Optional[float] = None
    product_key: Optional[str] = None  # Product API key

    def validate_and_clean(self) -> Dict[str, Any]:
        """Validate and clean all fields"""
        # Validate required fields
        if not self.type or self.type.strip() == "":
            raise HTTPException(status_code=400, detail="Type is required")
        
        if not self.message or self.message.strip() == "":
            raise HTTPException(status_code=400, detail="Message is required")
        
        # Validate type
        valid_types = ["bug", "feature", "suggestion"]
        if self.type not in valid_types:
            raise HTTPException(status_code=400, detail=f"Type must be one of: {', '.join(valid_types)}")
        
        # Clean and validate fields
        cleaned = {
            "type": self.type,
            "product": sanitize_text(self.product, 100) if self.product else None,
            "message": sanitize_text(self.message, 2000),
            "email": validate_email_address(self.email) if self.email else None
        }
        
        # Validate message length
        if not cleaned["message"]:
            raise HTTPException(status_code=400, detail="Message cannot be empty")
        
        return cleaned