(function() {
    'use strict';
    
    // Widget configuration
    const WIDGET_CONFIG = {
        apiUrl: window.location.origin + '/api/feedback',
        formProtectUrl: 'https://forms.blunek.services',
        position: 'bottom-right',
        theme: 'dark'
    };
    
    // Prevent multiple initializations
    if (window.BlunekFeedbackWidget) {
        return;
    }
    
    // Widget class
    class BlunekFeedbackWidget {
        constructor() {
            this.isOpen = false;
            this.renderTime = Date.now();
            this.formProtectToken = null;
            this.productKey = null;
            this.getProductKey();
            this.init();
        }
        
        getProductKey() {
            // Check for product key in script tag data attribute
            const scripts = document.querySelectorAll('script[src*="widget.js"]');
            for (let script of scripts) {
                const key = script.getAttribute('data-product-key');
                if (key) {
                    this.productKey = key;
                }
                
                // Check for encryption key
                const encKey = script.getAttribute('data-encryption-key');
                if (encKey) {
                    this.encryptionKey = encKey;
                }
                
                // Check for no-email mode
                const noEmail = script.getAttribute('data-no-email');
                if (noEmail === 'true') {
                    this.noEmailMode = true;
                }
                
                if (key) return;
            }
            
            // Check for product key in widget div data attribute
            const widgetDiv = document.getElementById('feedback-widget');
            if (widgetDiv) {
                const key = widgetDiv.getAttribute('data-product-key');
                if (key) {
                    this.productKey = key;
                }
                
                // Check for encryption key
                const encKey = widgetDiv.getAttribute('data-encryption-key');
                if (encKey) {
                    this.encryptionKey = encKey;
                }
                
                // Check for no-email mode
                const noEmail = widgetDiv.getAttribute('data-no-email');
                if (noEmail === 'true') {
                    this.noEmailMode = true;
                }
            }
        }
        
        init() {
            this.injectStyles();
            this.createButton();
            this.createModal();
            this.loadFormProtect();
        }
        
        injectStyles() {
            const styles = `
                .blunek-feedback-btn {
                    position: fixed;
                    bottom: 20px;
                    right: 20px;
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    color: white;
                    border: none;
                    border-radius: 50px;
                    padding: 12px 20px;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                    font-size: 14px;
                    font-weight: 500;
                    cursor: pointer;
                    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
                    transition: all 0.3s ease;
                    z-index: 999999;
                    user-select: none;
                    backdrop-filter: blur(10px);
                }
                
                .blunek-feedback-btn:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 6px 25px rgba(0, 0, 0, 0.2);
                    background: linear-gradient(135deg, #5a6fd8 0%, #6a4190 100%);
                }
                
                .blunek-feedback-modal {
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0, 0, 0, 0.7);
                    z-index: 1000000;
                    display: none;
                    align-items: center;
                    justify-content: center;
                    backdrop-filter: blur(5px);
                }
                
                .blunek-feedback-modal.open {
                    display: flex;
                    animation: fadeIn 0.3s ease;
                }
                
                .blunek-feedback-content {
                    background: #1a1a1a;
                    border-radius: 12px;
                    padding: 30px;
                    width: 90%;
                    max-width: 500px;
                    max-height: 80vh;
                    overflow-y: auto;
                    position: relative;
                    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.5);
                    animation: slideIn 0.3s ease;
                }
                
                .blunek-feedback-header {
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    margin-bottom: 20px;
                }
                
                .blunek-feedback-title {
                    color: #ffffff;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                    font-size: 20px;
                    font-weight: 600;
                    margin: 0;
                }
                
                .blunek-feedback-close {
                    background: none;
                    border: none;
                    color: #888;
                    font-size: 24px;
                    cursor: pointer;
                    padding: 0;
                    width: 30px;
                    height: 30px;
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    transition: all 0.2s ease;
                }
                
                .blunek-feedback-close:hover {
                    background: #333;
                    color: #fff;
                }
                
                .blunek-feedback-form {
                    color: #fff;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                }
                
                .blunek-feedback-group {
                    margin-bottom: 20px;
                }
                
                .blunek-feedback-label {
                    display: block;
                    margin-bottom: 8px;
                    color: #e0e0e0;
                    font-weight: 500;
                    font-size: 14px;
                }
                
                .blunek-feedback-required {
                    color: #ff6b6b;
                }
                
                .blunek-feedback-select,
                .blunek-feedback-input,
                .blunek-feedback-textarea {
                    width: 100%;
                    padding: 12px 16px;
                    border: 2px solid #333;
                    border-radius: 8px;
                    background: #2a2a2a;
                    color: #fff;
                    font-family: inherit;
                    font-size: 14px;
                    transition: border-color 0.2s ease;
                    box-sizing: border-box;
                }
                
                .blunek-feedback-select:focus,
                .blunek-feedback-input:focus,
                .blunek-feedback-textarea:focus {
                    outline: none;
                    border-color: #667eea;
                    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
                }
                
                .blunek-feedback-textarea {
                    resize: vertical;
                    min-height: 120px;
                }
                
                .blunek-feedback-counter {
                    text-align: right;
                    font-size: 12px;
                    color: #888;
                    margin-top: 5px;
                }
                
                .blunek-feedback-counter.warning {
                    color: #ffa500;
                }
                
                .blunek-feedback-counter.danger {
                    color: #ff6b6b;
                }
                
                .blunek-feedback-submit {
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    color: white;
                    border: none;
                    padding: 14px 28px;
                    border-radius: 8px;
                    font-family: inherit;
                    font-size: 14px;
                    font-weight: 600;
                    cursor: pointer;
                    transition: all 0.3s ease;
                    width: 100%;
                }
                
                .blunek-feedback-submit:hover:not(:disabled) {
                    background: linear-gradient(135deg, #5a6fd8 0%, #6a4190 100%);
                    transform: translateY(-1px);
                }
                
                .blunek-feedback-submit:disabled {
                    opacity: 0.6;
                    cursor: not-allowed;
                    transform: none;
                }
                
                .blunek-feedback-message {
                    padding: 12px 16px;
                    border-radius: 8px;
                    margin-bottom: 20px;
                    font-size: 14px;
                }
                
                .blunek-feedback-message.success {
                    background: rgba(76, 175, 80, 0.2);
                    border: 1px solid rgba(76, 175, 80, 0.5);
                    color: #4caf50;
                }
                
                .blunek-feedback-message.error {
                    background: rgba(244, 67, 54, 0.2);
                    border: 1px solid rgba(244, 67, 54, 0.5);
                    color: #f44336;
                }
                
                .blunek-honeypot {
                    position: absolute;
                    left: -9999px;
                    opacity: 0;
                    pointer-events: none;
                }
                
                @keyframes fadeIn {
                    from { opacity: 0; }
                    to { opacity: 1; }
                }
                
                @keyframes slideIn {
                    from { 
                        transform: translateY(-30px) scale(0.95); 
                        opacity: 0; 
                    }
                    to { 
                        transform: translateY(0) scale(1); 
                        opacity: 1; 
                    }
                }
                
                @media (max-width: 600px) {
                    .blunek-feedback-content {
                        width: 95%;
                        padding: 20px;
                        margin: 20px;
                    }
                    
                    .blunek-feedback-btn {
                        bottom: 15px;
                        right: 15px;
                        padding: 10px 16px;
                        font-size: 13px;
                    }
                }
            `;
            
            const styleSheet = document.createElement('style');
            styleSheet.textContent = styles;
            document.head.appendChild(styleSheet);
        }
        
        createButton() {
            this.button = document.createElement('button');
            this.button.className = 'blunek-feedback-btn';
            this.button.textContent = '💭 Feedback';
            this.button.onclick = () => this.openModal();
            document.body.appendChild(this.button);
        }
        
        createModal() {
            this.modal = document.createElement('div');
            this.modal.className = 'blunek-feedback-modal';
            this.modal.onclick = (e) => {
                if (e.target === this.modal) {
                    this.closeModal();
                }
            };
            
            this.modal.innerHTML = `
                <div class="blunek-feedback-content">
                    <div class="blunek-feedback-header">
                        <h2 class="blunek-feedback-title">Send Feedback</h2>
                        <button class="blunek-feedback-close" onclick="window.BlunekFeedbackWidget.closeModal()">×</button>
                    </div>
                    
                    <div id="feedback-message"></div>
                    
                    <form class="blunek-feedback-form" id="feedback-form">
                        <div class="blunek-feedback-group">
                            <label class="blunek-feedback-label">
                                Type <span class="blunek-feedback-required">*</span>
                            </label>
                            <select class="blunek-feedback-select" name="type" required>
                                <option value="">Select feedback type...</option>
                                <option value="bug">🐛 Bug Report</option>
                                <option value="feature">✨ Feature Request</option>
                                <option value="suggestion">💡 Suggestion</option>
                            </select>
                        </div>
                        
                        <div class="blunek-feedback-group">
                            <label class="blunek-feedback-label">Product</label>
                            <input 
                                type="text" 
                                class="blunek-feedback-input" 
                                name="product" 
                                placeholder="Which blünek product? (optional)"
                                maxlength="100"
                            >
                        </div>
                        
                        <div class="blunek-feedback-group">
                            <label class="blunek-feedback-label">
                                Message <span class="blunek-feedback-required">*</span>
                            </label>
                            <textarea 
                                class="blunek-feedback-textarea" 
                                name="message" 
                                placeholder="Tell us about your experience, report a bug, or suggest an improvement..."
                                maxlength="2000"
                                required
                            ></textarea>
                            <div class="blunek-feedback-counter">
                                <span id="char-count">0</span>/2000
                            </div>
                        </div>
                        
                        ${this.noEmailMode ? `
                        <div class="blunek-feedback-group">
                            <small style="color: #888; font-style: italic;">
                                Anonymous mode: No email collection for maximum privacy
                            </small>
                        </div>
                        ` : `
                        <div class="blunek-feedback-group">
                            <label class="blunek-feedback-label">Email (for follow-up)</label>
                            <input 
                                type="email" 
                                class="blunek-feedback-input" 
                                name="email" 
                                placeholder="your@email.com (optional)"
                                maxlength="255"
                            >
                        </div>
                        `}
                        
                        <!-- Honeypot field -->
                        <input type="text" name="honeypot" class="blunek-honeypot" autocomplete="off" tabindex="-1">
                        
                        <button type="submit" class="blunek-feedback-submit" id="submit-btn">
                            Send Feedback
                        </button>
                    </form>
                </div>
            `;
            
            document.body.appendChild(this.modal);
            this.setupFormHandlers();
        }
        
        setupFormHandlers() {
            const form = document.getElementById('feedback-form');
            const messageTextarea = form.querySelector('[name="message"]');
            const charCount = document.getElementById('char-count');
            const submitBtn = document.getElementById('submit-btn');
            
            // Character counter
            messageTextarea.addEventListener('input', () => {
                const count = messageTextarea.value.length;
                charCount.textContent = count;
                
                const counter = charCount.parentElement;
                counter.className = 'blunek-feedback-counter';
                if (count > 1800) counter.classList.add('danger');
                else if (count > 1600) counter.classList.add('warning');
            });
            
            // Form submission
            form.addEventListener('submit', (e) => this.handleSubmit(e));
        }
        
        async loadFormProtect() {
            try {
                const response = await fetch(`${WIDGET_CONFIG.formProtectUrl}/token`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ domain: window.location.hostname })
                });
                
                if (response.ok) {
                    const data = await response.json();
                    this.formProtectToken = data.token;
                }
            } catch (error) {
                console.warn('FormProtect not available:', error);
            }
        }
        
        // Client-side encryption using Web Crypto API
        async deriveKey(password, salt) {
            const encoder = new TextEncoder();
            const keyMaterial = await window.crypto.subtle.importKey(
                'raw',
                encoder.encode(password),
                { name: 'PBKDF2' },
                false,
                ['deriveKey']
            );
            
            return window.crypto.subtle.deriveKey(
                {
                    name: 'PBKDF2',
                    salt: salt,
                    iterations: 100000,
                    hash: 'SHA-256'
                },
                keyMaterial,
                { name: 'AES-GCM', length: 256 },
                false,
                ['encrypt']
            );
        }
        
        async encryptData(data, encryptionKey) {
            if (!window.crypto || !window.crypto.subtle) {
                throw new Error('Web Crypto API not supported');
            }
            
            try {
                const encoder = new TextEncoder();
                const salt = window.crypto.getRandomValues(new Uint8Array(16));
                const iv = window.crypto.getRandomValues(new Uint8Array(12));
                
                // Derive key from password
                const key = await this.deriveKey(encryptionKey, salt);
                
                // Encrypt the data
                const encryptedData = await window.crypto.subtle.encrypt(
                    {
                        name: 'AES-GCM',
                        iv: iv
                    },
                    key,
                    encoder.encode(data)
                );
                
                // Combine salt, iv, and encrypted data
                const combined = new Uint8Array(salt.length + iv.length + encryptedData.byteLength);
                combined.set(salt, 0);
                combined.set(iv, salt.length);
                combined.set(new Uint8Array(encryptedData), salt.length + iv.length);
                
                // Return base64 encoded with prefix
                return 'CLIENT_ENC:' + btoa(String.fromCharCode(...combined));
            } catch (error) {
                console.error('Encryption failed:', error);
                throw new Error('Failed to encrypt data');
            }
        }
        
        async handleSubmit(e) {
            e.preventDefault();
            
            const form = e.target;
            const submitBtn = document.getElementById('submit-btn');
            const messageDiv = document.getElementById('feedback-message');
            
            // Disable submit button
            submitBtn.disabled = true;
            submitBtn.textContent = 'Sending...';
            
            // Clear previous messages
            messageDiv.innerHTML = '';
            
            // Get form data
            const formData = new FormData(form);
            let message = formData.get('message');
            let email = formData.get('email');
            
            // Encrypt data if encryption key is provided
            if (this.encryptionKey) {
                try {
                    message = await this.encryptData(message, this.encryptionKey);
                    if (email) {
                        email = await this.encryptData(email, this.encryptionKey);
                    }
                } catch (error) {
                    this.showMessage('Encryption failed. Please try again.', 'error');
                    submitBtn.disabled = false;
                    submitBtn.textContent = 'Send Feedback';
                    return;
                }
            }
            
            const data = {
                type: formData.get('type'),
                product: formData.get('product'),
                message: message,
                email: email,
                honeypot: formData.get('honeypot'),
                formprotect_token: this.formProtectToken,
                render_time: this.renderTime / 1000,
                product_key: this.productKey
            };
            
            try {
                const response = await fetch(WIDGET_CONFIG.apiUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (response.ok) {
                    this.showMessage('Thank you for your feedback! We appreciate your input.', 'success');
                    form.reset();
                    document.getElementById('char-count').textContent = '0';
                    
                    // Close modal after delay
                    setTimeout(() => this.closeModal(), 2000);
                } else {
                    this.showMessage(result.detail || 'Failed to send feedback. Please try again.', 'error');
                }
            } catch (error) {
                this.showMessage('Network error. Please check your connection and try again.', 'error');
            } finally {
                // Re-enable submit button
                submitBtn.disabled = false;
                submitBtn.textContent = 'Send Feedback';
            }
        }
        
        showMessage(text, type) {
            const messageDiv = document.getElementById('feedback-message');
            messageDiv.innerHTML = `<div class="blunek-feedback-message ${type}">${text}</div>`;
        }
        
        openModal() {
            this.renderTime = Date.now(); // Reset render time
            this.modal.classList.add('open');
            this.isOpen = true;
            
            // Focus first input
            setTimeout(() => {
                const firstInput = this.modal.querySelector('select[name="type"]');
                if (firstInput) firstInput.focus();
            }, 300);
        }
        
        closeModal() {
            this.modal.classList.remove('open');
            this.isOpen = false;
            
            // Clear form and messages
            const form = document.getElementById('feedback-form');
            const messageDiv = document.getElementById('feedback-message');
            form.reset();
            messageDiv.innerHTML = '';
            document.getElementById('char-count').textContent = '0';
        }
    }
    
    // Initialize widget when DOM is ready
    function initWidget() {
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => {
                window.BlunekFeedbackWidget = new BlunekFeedbackWidget();
            });
        } else {
            window.BlunekFeedbackWidget = new BlunekFeedbackWidget();
        }
    }
    
    // Keyboard shortcuts
    document.addEventListener('keydown', (e) => {
        if (window.BlunekFeedbackWidget) {
            if (e.key === 'Escape' && window.BlunekFeedbackWidget.isOpen) {
                window.BlunekFeedbackWidget.closeModal();
            }
        }
    });
    
    initWidget();
})();