#!/usr/bin/env python3
"""
Webhook delivery system for blünek Feedback System
Handles reliable delivery with retry logic
"""

import asyncio
import aiohttp
import json
import logging
from datetime import datetime
from typing import Dict, Optional
from sqlalchemy.orm import Session

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


import ipaddress
from urllib.parse import urlparse

def is_safe_webhook_url(url: str) -> bool:
    """
    Validate webhook URL to prevent SSRF attacks
    Blocks: localhost, private IPs, internal networks
    """
    if not url:
        return False
    
    try:
        parsed = urlparse(url)
        
        # Must be http or https
        if parsed.scheme not in ('http', 'https'):
            return False
        
        hostname = parsed.hostname
        if not hostname:
            return False
        
        # Block localhost variations
        blocked_hosts = ['localhost', '127.0.0.1', '0.0.0.0', '::1', '[::1]']
        if hostname.lower() in blocked_hosts:
            return False
        
        # Block internal hostnames
        if hostname.endswith('.local') or hostname.endswith('.internal'):
            return False
        
        # Try to resolve and check if IP is private
        try:
            ip = ipaddress.ip_address(hostname)
            if ip.is_private or ip.is_loopback or ip.is_reserved or ip.is_link_local:
                return False
        except ValueError:
            # It's a hostname, not an IP - that's OK
            # DNS resolution could still resolve to private IP, but 
            # we can't easily check that without making a request
            pass
        
        # Block common internal ports
        if parsed.port and parsed.port in [22, 23, 25, 3306, 5432, 6379, 27017]:
            return False
        
        return True
        
    except Exception:
        return False


class WebhookDelivery:
    def __init__(self):
        self.retry_attempts = 3
        self.timeout = 10  # seconds
        
    async def send_webhook(
        self, 
        webhook_url: str, 
        feedback_data: Dict,
        product_name: str,
        retry_count: int = 0
    ) -> bool:
        """
        Send webhook with retry logic
        Returns True if successful, False otherwise
        """
        if not webhook_url or not is_safe_webhook_url(webhook_url):
            logger.warning(f"Invalid or unsafe webhook URL: {webhook_url}")
            return False
            
        payload = {
            "type": feedback_data.get("type"),
            "message": feedback_data.get("message"),
            "email": feedback_data.get("email"),
            "product": product_name,
            "timestamp": datetime.utcnow().isoformat() + "Z"
        }
        
        headers = {
            "Content-Type": "application/json",
            "User-Agent": "Blunek-Feedback-System/1.0"
        }
        
        try:
            async with aiohttp.ClientSession(timeout=aiohttp.ClientTimeout(total=self.timeout)) as session:
                async with session.post(webhook_url, json=payload, headers=headers) as response:
                    if 200 <= response.status < 300:
                        logger.info(f"Webhook delivered successfully to {webhook_url} (status: {response.status})")
                        return True
                    else:
                        logger.warning(f"Webhook failed with status {response.status} to {webhook_url}")
                        
        except asyncio.TimeoutError:
            logger.warning(f"Webhook timeout to {webhook_url}")
        except aiohttp.ClientError as e:
            logger.warning(f"Webhook client error to {webhook_url}: {e}")
        except Exception as e:
            logger.error(f"Webhook unexpected error to {webhook_url}: {e}")
        
        # Retry logic
        if retry_count < self.retry_attempts - 1:
            retry_delay = 2 ** retry_count  # Exponential backoff
            logger.info(f"Retrying webhook to {webhook_url} in {retry_delay} seconds (attempt {retry_count + 2})")
            await asyncio.sleep(retry_delay)
            return await self.send_webhook(webhook_url, feedback_data, product_name, retry_count + 1)
        
        logger.error(f"Webhook delivery failed after {self.retry_attempts} attempts to {webhook_url}")
        return False

    async def deliver_feedback_webhook(self, product, feedback_data: Dict) -> bool:
        """
        Deliver webhook for feedback submission
        """
        if not product or not product.webhook_url:
            return False
            
        return await self.send_webhook(
            product.webhook_url,
            feedback_data,
            product.name
        )

# Global webhook delivery instance
webhook_delivery = WebhookDelivery()